<?php
include 'db_config.php'; // Includes session_start() and $pdo

// If a user is already logged in, redirect them away from the signup page
if (isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$error = '';
$success = '';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];

    // 1. Basic Validation
    if (empty($username) || empty($password) || empty($confirm_password)) {
        $error = "All fields are required.";
    } elseif ($password !== $confirm_password) {
        $error = "Passwords do not match.";
    } elseif (strlen($password) < 8) {
        $error = "Password must be at least 8 characters long.";
    } else {
        // 2. Check if username already exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->execute([$username]);
        
        if ($stmt->fetch()) {
            $error = "Username already taken. Please choose another one.";
        } else {
            // 3. Hash the password securely
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            // 4. Insert the new user
            try {
                $stmt = $pdo->prepare("INSERT INTO users (username, password_hash) VALUES (?, ?)");
                $stmt->execute([$username, $password_hash]);
                
                $success = "Registration successful! You can now **login**.";
                // Clear the form data after successful registration
                unset($username); 
            } catch (\PDOException $e) {
                // Log the error for debugging but provide a generic message to the user
                error_log("User registration error: " . $e->getMessage());
                $error = "An unexpected error occurred during registration. Please try again.";
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Sign Up | Invoice System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<div class="container d-flex justify-content-center align-items-center" style="min-height: 100vh;">
    <div class="card p-4 shadow-sm" style="width: 450px;">
        <h2 class="card-title text-center mb-4">Register New User</h2>
        
        <?php if ($error): ?>
            <div class="alert alert-danger" role="alert"><?= $error ?></div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success" role="alert">
                <?= $success ?> 
                <a href="login.php" class="alert-link">Click here to log in.</a>
            </div>
        <?php endif; ?>

        <form method="POST">
            <div class="mb-3">
                <label for="username" class="form-label">Username</label>
                <input type="text" class="form-control" id="username" name="username" value="<?= htmlspecialchars($username ?? '') ?>" required>
            </div>
            <div class="mb-3">
                <label for="password" class="form-label">Password (Min 8 characters)</label>
                <input type="password" class="form-control" id="password" name="password" required>
            </div>
            <div class="mb-4">
                <label for="confirm_password" class="form-label">Confirm Password</label>
                <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
            </div>
            <button type="submit" class="btn btn-success w-100">Sign Up</button>
        </form>
        
        <p class="text-center mt-3 mb-0">
            Already have an account? <a href="login.php">Log in here</a>
        </p>
    </div>
</div>
</body>
</html>