<?php
// callback.php

// Error handling and logging
ini_set('display_errors', 0);
error_reporting(0);
ini_set('log_errors', 1);
ini_set('error_log', 'callback_error.log');

// Include DB configuration
include_once 'db_config.php';

// Set the content type to JSON
header('Content-Type: application/json');

// Log file for debugging
$debug_log = 'callback_debug.log';
function log_debug($msg) {
    global $debug_log;
    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - $msg\n", FILE_APPEND);
}

// Retrieve the transaction details from the GET parameters
$transaction_id = $_GET['transaction_id'] ?? null;
$status = $_GET['status'] ?? null;
$message = $_GET['message'] ?? null;

log_debug("Received callback: Transaction ID: $transaction_id, Status: $status, Message: $message");

if (!$transaction_id || !$status) {
    log_debug("Error: Missing transaction_id or status.");
    echo json_encode(['success' => false, 'message' => 'Invalid request parameters.']);
    exit;
}

// Check if payment was successful
if (strtoupper($status) === 'SUCCESS') {
    // Derive invoice number from transaction ID by removing the last 6 digits
    $invoice_number = substr($transaction_id, 0, -6);

    if (empty($invoice_number)) {
        log_debug("Error: Could not derive invoice number from transaction ID: $transaction_id");
        echo json_encode(['success' => false, 'message' => 'Could not derive invoice number.']);
        exit;
    }
    
    log_debug("Derived invoice number: $invoice_number");

    try {
        if (!isset($pdo)) {
             throw new Exception("Database connection is not available.");
        }

        // Update invoice status to 'Paid'
        $stmt = $pdo->prepare("UPDATE invoices SET status = 'Paid', paid_date = CURDATE() WHERE invoice_number = ?");
        $stmt->execute([$invoice_number]);
        
        $affected_rows = $stmt->rowCount();

        if ($affected_rows > 0) {
            log_debug("Successfully updated invoice '$invoice_number' to 'Paid'.");

            // Attempt to send a confirmation SMS
            include_once 'send_sms.php';

            // Get the client's phone number and invoice total
            $phone_stmt = $pdo->prepare(
                "SELECT c.phone_number, i.total 
                 FROM clients c 
                 JOIN invoices i ON c.client_id = i.client_id 
                 WHERE i.invoice_number = ?"
            );
            $phone_stmt->execute([$invoice_number]);
            $result = $phone_stmt->fetch(PDO::FETCH_ASSOC);

            if ($result && !empty($result['phone_number'])) {
                $phone_number = $result['phone_number'];
                $amount_paid = $result['total'];
                $message = "Your payment of GHS {$amount_paid} for invoice {$invoice_number} was successful. Thank you for using Krola Services.";
                
                if (send_sms($phone_number, $message)) {
                    log_debug("Successfully sent confirmation SMS to {$phone_number}.");
                } else {
                    log_debug("Failed to send confirmation SMS to {$phone_number}.");
                }
            } else {
                log_debug("Could not find phone number for invoice '{$invoice_number}' to send SMS.");
            }

            echo json_encode(['success' => true, 'message' => 'Invoice status updated.']);
        } else {
            log_debug("Warning: No invoice found with invoice number '$invoice_number' to update.");
            echo json_encode(['success' => false, 'message' => 'Invoice not found.']);
        }

    } catch (Exception $e) {
        log_debug("Database Error: " . $e->getMessage());
        // In a production environment, you might want a more generic error message.
        echo json_encode(['success' => false, 'message' => 'An internal error occurred.']);
    }

} else {
    // Handle other statuses if needed, e.g., 'FAILED'
    log_debug("Transaction status was not 'SUCCESS'. Status: $status");
    echo json_encode(['success' => true, 'message' => 'Callback received but no action taken for this status.']);
}
?>
