<?php
// Bills USSD Application

// Error handling: prevent raw output, which breaks USSD JSON response
ini_set('display_errors', 0);
error_reporting(0);

include_once 'db_config.php';

// Gracefully handle DB connection failure
if (!isset($pdo)) {
    $response_msg = "Service is temporarily unavailable. Please try again later.";
    $response = ["message" => $response_msg, "reply" => false];
    header('Content-type: application/json');
    echo json_encode($response);
    exit;
}

// Read incoming JSON request
$request = file_get_contents("php://input");
$data = json_decode($request, true);

// Extract parameters
$session_id = $data['sessionId'] ?? '';
$is_new_session = $data['new'] ?? false;
$msisdn = $data['msisdn'] ?? '';
$user_data = $data['message'] ?? '';

// Format phone number for the current user. Assume MSISDN is in '233...' format.
$phone = '+233' . substr($msisdn, 3);

// Start session
session_id($session_id);
if ($is_new_session) {
    if (session_status() === PHP_SESSION_ACTIVE) {
        session_destroy();
    }
    session_start();
    $_SESSION['menu'] = 'main_menu';
    $_SESSION['phone'] = $phone; // The number of the person dialing
    $_SESSION['reply'] = true;
} else {
    session_start();
}

// Function to fetch and format invoices
function get_invoices_for_phone($pdo, $phone_number) {
    $stmt = $pdo->prepare("SELECT i.invoice_id, i.invoice_number, i.total, i.status FROM invoices i JOIN clients c ON i.client_id = c.client_id WHERE c.phone_number = ? AND i.status = 'Pending'");
    $stmt->execute([$phone_number]);
    return $stmt->fetchAll();
}

// Main USSD logic
switch ($_SESSION['menu'] ?? 'main_menu') {
    case 'main_menu':
    default:
        $response_msg = "Welcome. Please select an option:\n1. Pay Bills\n2. My Pending Invoices";
        // Only advance state for POST requests (real USSD interactions)
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $_SESSION['menu'] = 'process_main_menu';
        } else {
            $_SESSION['menu'] = 'main_menu';
        }
        $_SESSION['reply'] = true;
        break;

    case 'process_main_menu':
        if ($user_data === '1') { // Pay Bills
            $_SESSION['menu'] = 'ask_for_phone';
            $response_msg = "Please enter the phone number you want to pay for:";
            $_SESSION['reply'] = true;
        } elseif ($user_data === '2') { // My Pending Invoices
            $invoices = get_invoices_for_phone($pdo, $_SESSION['phone']);
            if (count($invoices) > 0) {
                $response_msg = "Your pending invoices:\n";
                $_SESSION['invoices'] = [];
                foreach ($invoices as $index => $invoice) {
                    $response_msg .= ($index + 1) . ". " . $invoice['invoice_number'] . " - GHS " . $invoice['total'] . "\n";
                    $_SESSION['invoices'][$index + 1] = $invoice;
                }
                $response_msg .= "\nEnter invoice number to pay.";
                $_SESSION['menu'] = 'select_invoice_to_pay';
            } else {
                $response_msg = "You have no pending invoices.";
                $_SESSION['reply'] = false; // End the session
            }
        } else {
            $response_msg = "Invalid option. Please select:\n1. Pay Bills\n2. My Pending Invoices";
            $_SESSION['menu'] = 'process_main_menu';
        }
        break;

    case 'ask_for_phone':
        // User has entered the phone number
        $entered_phone = trim($user_data);
        $phone_to_query = null;

        // Handle numbers where leading 0 is stripped (9 digits like 543177429)
        if (preg_match('/^[1-9][0-9]{8}$/', $entered_phone)) {
            $phone_to_query = '+233' . $entered_phone;
        } 
        // Handle 10-digit numbers starting with 0 (like 0543177429)
        elseif (preg_match('/^0[0-9]{9}$/', $entered_phone)) {
            $phone_to_query = '+233' . substr($entered_phone, 1);
        }

        if ($phone_to_query) {
            $_SESSION['payment_phone'] = $phone_to_query; // Store the formatted number
            $invoices = get_invoices_for_phone($pdo, $phone_to_query);

            if (count($invoices) > 0) {
                $response_msg = "Pending invoices for " . $entered_phone . ":\n";
                $_SESSION['invoices'] = [];
                foreach ($invoices as $index => $invoice) {
                    $response_msg .= ($index + 1) . ". " . $invoice['invoice_number'] . " - GHS " . $invoice['total'] . "\n";
                    $_SESSION['invoices'][$index + 1] = $invoice;
                }
                $response_msg .= "\nEnter invoice number to pay.";
                $_SESSION['menu'] = 'select_invoice_to_pay';
            } else {
                $response_msg = "No pending invoices found for " . $entered_phone;
                $_SESSION['reply'] = false; // End the session
            }
        } else {
            $debug_info = "Received: " . $entered_phone;
            $response_msg = "Invalid phone number. " . $debug_info . ". Please enter a valid 10-digit Ghanaian number.";
            $_SESSION['menu'] = 'ask_for_phone'; // Ask again
        }
        break;

    case 'select_invoice_to_pay':
        $selected_option = (int)$user_data;
        if (isset($_SESSION['invoices'][$selected_option])) {
            $selected_invoice = $_SESSION['invoices'][$selected_option];
            $_SESSION['selected_invoice'] = $selected_invoice;
            $response_msg = "Pay Invoice " . $selected_invoice['invoice_number'] . " (GHS " . $selected_invoice['total'] . ")?\n1. Confirm\n2. Cancel";
            $_SESSION['menu'] = 'confirm_payment';
        } else {
            $response_msg = "Invalid selection. Please choose a valid invoice number from the list.";
        }
        break;

    case 'confirm_payment':
        if ($user_data === '1') {
            // User confirmed payment
            $selected_invoice = $_SESSION['selected_invoice'];
            
            // Format phone number: replace +233 with 0
            $customer_number = '0' . substr($_SESSION['phone'], 4);

            $postData = [
                'customer_number' => $customer_number,
                'amount' => $selected_invoice['total'],
                'transaction_id' => $selected_invoice['invoice_number'],
                'network_code' => 'MTN'
            ];
            
            $initiate_url = 'https://krolatech.org/initiate_payment_bills.php';
            
            // Use cURL to post to the specified URL
            $ch = curl_init($initiate_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postData));
            curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
            curl_exec($ch);
            curl_close($ch);
            
            $response_msg = "Payment initiated. You will receive a prompt to complete the payment.";
            $_SESSION['reply'] = false; // End the session

        } elseif ($user_data === '2') {
            $response_msg = "Payment cancelled.";
            $_SESSION['reply'] = false; // End the session
        } else {
            $response_msg = "Invalid option. Please enter 1 to confirm or 2 to cancel.";
        }
        break;

}
// Prepare response
$response = [
    "message" => $response_msg,
    "reply" => $_SESSION['reply'] ?? true
];

header('Content-type: application/json');
echo json_encode($response);
?>